<?php
session_start();
require_once '../config/db.php';
require_once '../includes/functions.php';
require_once '../auth/auth.php';

$error = '';
$success = '';

// Get store settings
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_logo', 'primary_color', 'secondary_color', 'store_address', 'store_phone', 'store_email')");
    $stmt->execute();
    $storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    $storeName = $storeSettings['store_name'] ?? 'POS System';
    $storeLogo = $storeSettings['store_logo'] ?? 'assets/img/logo.png';
    $primaryColor = $storeSettings['primary_color'] ?? '#007bff';
    $secondaryColor = $storeSettings['secondary_color'] ?? '#0056b3';
    $storeAddress = $storeSettings['store_address'] ?? '';
    $storePhone = $storeSettings['store_phone'] ?? '';
    $storeEmail = $storeSettings['store_email'] ?? '';
} catch (PDOException $e) {
    error_log("Error fetching store settings: " . $e->getMessage());
    $storeName = 'POS System';
    $storeLogo = 'assets/img/logo.png';
    $primaryColor = '#007bff';
    $secondaryColor = '#0056b3';
}

// Check for timeout parameter
if (isset($_GET['timeout']) && $_GET['timeout'] == '1') {
    $error = 'Your session has expired due to inactivity. Please log in again.';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $remember = isset($_POST['remember']) ? true : false;

    try {
        // First try to find user by username
        $stmt = $pdo->prepare('SELECT u.*, r.name as role FROM users u JOIN roles r ON u.role_id = r.id WHERE u.username = :username');
        $stmt->execute(['username' => $username]);
        $user = $stmt->fetch();

        // If not found by username and input looks like an email, try email
        if (!$user && filter_var($username, FILTER_VALIDATE_EMAIL)) {
            $stmt = $pdo->prepare('SELECT u.*, r.name as role FROM users u JOIN roles r ON u.role_id = r.id WHERE u.email = :email');
            $stmt->execute(['email' => $username]);
            $user = $stmt->fetch();
        }

        if ($user) {
            if (password_verify($password, $user['password'])) {
                if (isset($user['enabled']) && $user['enabled'] == 0) {
                    $error = "Your account is disabled. Please contact the administrator.";
                    // Optionally log the attempt
                    $stmt = $pdo->prepare('INSERT INTO system_logs (user_id, action, details) VALUES (?, ?, ?)');
                    $stmt->execute([$user['id'], 'login_failed', 'Attempted login to disabled account']);
                } else {
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['role'] = $user['role'];
                    
                    // Set remember me cookie if requested
                    if ($remember) {
                        $token = bin2hex(random_bytes(32));
                        $expires = time() + (86400 * 30); // 30 days
                        
                        // Store token in database
                        $stmt = $pdo->prepare('INSERT INTO remember_tokens (user_id, token, expires_at) VALUES (?, ?, ?)');
                        $stmt->execute([$user['id'], $token, date('Y-m-d H:i:s', $expires)]);
                        
                        // Set cookie
                        setcookie('remember_token', $token, $expires, '/', '', true, true);
                    }
                    
                    // Log successful login
                    $stmt = $pdo->prepare('INSERT INTO system_logs (user_id, action, details) VALUES (?, ?, ?)');
                    $stmt->execute([$user['id'], 'login', 'User logged in successfully']);
                    
                    header('Location: ../pages/dashboard.php');
                    exit;
                }
            } else {
                $error = 'Invalid password.';
                // Log failed login attempt
                $stmt = $pdo->prepare('INSERT INTO system_logs (user_id, action, details) VALUES (?, ?, ?)');
                $stmt->execute([$user['id'], 'login_failed', 'Invalid password']);
            }
        } else {
            $error = 'User not found.';
        }
    } catch (PDOException $e) {
        error_log("Database error: " . $e->getMessage());
        $error = 'Database error. Please try again later.';
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - <?php echo htmlspecialchars($storeName); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: <?php echo $primaryColor; ?>;
            --secondary-color: <?php echo $secondaryColor; ?>;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background-color: #f8f9fa;
            margin: 0;
            padding: 0;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .login-container {
            width: 100%;
            max-width: 400px;
            padding: 20px;
        }
        
        .card {
            border: none;
            border-radius: 25px;
            box-shadow: 0 0 20px rgba(0, 0, 0, 0.1);
            overflow: visible;
        }
        
        .card-header {
            background: var(--primary-color);
            color: white;
            text-align: center;
            padding: 15px 20px 10px;
            position: relative;
            border: none;
            border-top-left-radius: 25px;
            border-top-right-radius: 25px;
        }
        
        .profile-photo {
            width: 100px;
            height: 100px;
            margin: 0 auto 10px;
            position: relative;
            overflow: visible;
        }
        
        .profile-photo img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            position: relative;
            z-index: 1;
        }
        
        .card-header h4 {
            color: white;
            margin: 0;
            font-weight: 600;
            font-size: 1.1rem;
        }
        
        .card-body {
            padding: 20px 30px 30px;
        }
        
        .form-control {
            border-radius: 8px;
            padding: 12px 15px;
            border: 1px solid #ddd;
            font-size: 14px;
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.1);
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            border: none;
            border-radius: 8px;
            padding: 12px;
            font-weight: 500;
            width: 100%;
            margin-top: 10px;
        }
        
        .btn-primary:hover {
            background-color: var(--secondary-color);
        }
        
        .form-check-input:checked {
            background-color: var(--primary-color);
            border-color: var(--primary-color);
        }
        
        .loading {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.9);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 9999;
        }
        
        .loading.active {
            display: flex;
        }
        
        .loading-spinner {
            width: 50px;
            height: 50px;
            border: 5px solid #f3f3f3;
            border-top: 5px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .alert {
            border-radius: 8px;
            padding: 12px 15px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert i {
            margin-right: 8px;
        }
        
        .form-text {
            font-size: 12px;
            color: #6c757d;
            margin-top: 5px;
        }
        
        .form-check {
            margin-top: 15px;
        }
        
        .form-check-label {
            font-size: 14px;
            color: #6c757d;
        }
        
        .mt-3 {
            margin-top: 1rem;
        }
        
        .text-center {
            text-align: center;
        }
        
        .small {
            font-size: 12px;
        }
        
        .text-muted {
            color: #6c757d !important;
        }
        
        .d-block {
            display: block;
        }
    </style>
</head>
<body>
    <div class="loading">
        <div class="loading-spinner"></div>
    </div>
    
    <div class="login-container">
        <div class="card">
            <div class="card-header">
                <div class="profile-photo">
                    <?php echo getStoreLogo('login-logo', 'width: 100%; height: 100%; object-fit: contain;'); ?>
                </div>
                <h4><?php echo htmlspecialchars($storeName); ?></h4>
            </div>
            <div class="card-body">
                <?php if ($error): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>
                
                <form method="post" id="loginForm">
                    <div class="mb-4">
                        <label for="username" class="form-label">Username or Email</label>
                        <div class="input-group">
                            <span class="input-group-text">
                                <i class="fas fa-user"></i>
                            </span>
                            <input type="text" class="form-control" id="username" name="username" required 
                                   placeholder="Enter your username or email" autocomplete="username">
                        </div>
                    </div>
                    
                    <div class="mb-4">
                        <label for="password" class="form-label">Password</label>
                        <div class="input-group">
                            <span class="input-group-text">
                                <i class="fas fa-lock"></i>
                            </span>
                            <input type="password" class="form-control" id="password" name="password" required 
                                   placeholder="Enter your password" autocomplete="current-password">
                            <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                <i class="fas fa-eye"></i>
                            </button>
                        </div>
                    </div>
                    
                    <div class="mb-4 form-check">
                        <input type="checkbox" class="form-check-input" id="remember" name="remember">
                        <label class="form-check-label" for="remember">Remember me</label>
                    </div>
                    
                    <button type="submit" class="btn btn-primary w-100 mb-3">
                        <i class="fas fa-sign-in-alt me-2"></i>Sign In
                    </button>
                    
                    <div class="text-center">
                        <a href="reset_password.php" class="forgot-password">
                            <i class="fas fa-key me-1"></i>Forgot Password?
                        </a>
                    </div>
                </form>
            </div>
        </div>
        
        
        <div class="footer">
            <?php include '../includes/footer.php'; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Toggle password visibility
            const togglePassword = document.getElementById('togglePassword');
            const password = document.getElementById('password');
            
            togglePassword.addEventListener('click', function() {
                const type = password.getAttribute('type') === 'password' ? 'text' : 'password';
                password.setAttribute('type', type);
                this.querySelector('i').classList.toggle('fa-eye');
                this.querySelector('i').classList.toggle('fa-eye-slash');
            });
            
            // Form submission with loading state
            const form = document.getElementById('loginForm');
            const loading = document.querySelector('.loading');
            
            form.addEventListener('submit', function() {
                loading.classList.add('active');
            });
            
            // Check for remember me cookie
            const rememberToken = getCookie('remember_token');
            if (rememberToken) {
                // Auto-login with token
                fetch('verify_token.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        token: rememberToken
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        window.location.href = '../pages/dashboard.php';
                    }
                })
                .catch(error => console.error('Error:', error));
            }
        });
        
        // Helper function to get cookie value
        function getCookie(name) {
            const value = `; ${document.cookie}`;
            const parts = value.split(`; ${name}=`);
            if (parts.length === 2) return parts.pop().split(';').shift();
        }
        
        // Hide loading spinner when page is loaded
        window.addEventListener('load', function() {
            document.querySelector('.loading').classList.remove('active');
        });
    </script>
</body>
</html>

